<?php
/**
 * Module Name: WPLMS Learning Management System for WordPress Integration
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

if ( defined( 'BP_COURSE_MOD_INSTALLED' ) ) :

class WPAchievements_Module_WPLMS {

  /**
   * Init required hooks
   *
   * @static
   * @access  public
   * @return  void
   */
  public static function init() {

    // Register settings hooks
    add_filter( 'wpachievements_get_sections_module', array( __CLASS__, 'register_section' ) );
    add_filter( 'wpachievements_achievements_modules_admin_settings', array( __CLASS__, 'admin_settings'), 10, 3 );
    add_action( 'wpachievements_achievements_event_details_footer', array( __CLASS__, 'achievements_event_details_footer'), 10, 4 );

    // Register WPAchievements hooks
    add_action( 'wpachievements_admin_events', array( __CLASS__, 'trigger_dropdown' ) );
    add_filter( 'wpachievements_trigger_description', array( __CLASS__, 'selected_trigger' ) );
    add_filter( 'wpachievements_activity_description', array( __CLASS__, 'activity_description'), 10, 4 );
    add_filter( 'wpachievements_quest_description', array( __CLASS__, 'quest_description' ), 10, 3 );

    add_filter( 'wpachievements_achievements_event_details_data_post_id_show', array( __CLASS__, 'event_details_data_post_id_show' ), 10, 3 );
    add_filter( 'wpachievements_achievements_event_details_data_post_id_text', array( __CLASS__, 'event_details_data_post_id_text' ), 10, 3 );

    add_filter( 'wpachievements_proceed_new_activity', array( __CLASS__, 'check_actvity'), 10, 5 );

    // Register WPLMS hooks
    // Quiz
    add_action('wplms_start_quiz', array( __CLASS__, 'start_quiz' ), 10, 2 );
    add_action('wplms_submit_quiz', array( __CLASS__, 'submit_quiz' ), 10, 3 );
    add_action('wplms_evaluate_quiz', array( __CLASS__, 'evaluate_quiz' ), 10, 4 );
    // Course
    add_action('wplms_start_course', array( __CLASS__, 'start_course' ), 10, 2 );
    add_action('wplms_submit_course', array( __CLASS__, 'submit_course' ), 10, 2 );
    add_action('wplms_evaluate_course', array( __CLASS__, 'evaluate_course' ), 10, 3 );
    add_action('wplms_course_subscribed', array( __CLASS__, 'course_subscribed' ), 10, 4 );
    add_action('wplms_course_reset', array( __CLASS__, 'course_reset'), 10, 2 );
    add_action('wplms_course_retake', array( __CLASS__, 'course_retake'), 10 , 2 );
    add_action('wplms_course_review', array( __CLASS__, 'course_review'), 10, 3 );
    add_action('wplms_course_unsubscribe',array( __CLASS__, 'course_unsubscribe'), 10, 3 );
    add_action('wplms_renew_course',array( __CLASS__, 'renew_course'), 10, 2 );
    add_action('wplms_unit_complete', array( __CLASS__, 'unit_complete' ), 10, 4 );
    // Extra
    add_action('wplms_certificate_earned',array( __CLASS__, 'certificate_earned'), 10, 4 );
    add_action('wplms_badge_earned', array( __CLASS__, 'badge_earned'),10,4);
    // Assignment
    add_action('wplms_submit_assignment', array( __CLASS__, 'submit_assignment' ), 10, 2 );
    add_action('wplms_start_assignment', array( __CLASS__, 'start_assignment' ), 10, 2 );
    add_action('wplms_evaluate_assignment', array( __CLASS__, 'evaluate_assignment' ), 10, 3 );

    //Member/Profile Output
    add_action( 'bp_directory_members_item', array( __CLASS__,'bp_profile_grid' ) );
    add_action( 'bp_before_profile_loop_content', array( __CLASS__,'bp_profile' ) );
  }

  /**
  * Show rank status and all achievements/wqests in profile member view
  *
  * @static
  * @access  public
  * @return
  */
  public static function bp_profile(){

    $profile_user = get_userdata( bp_displayed_user_id() );
    if ( 'Enable' == wpachievements_get_site_option('wpachievements_rank_status') ) {
      $add_ranktext = ' ( '.__('Rank', 'wpachievements').': '.wpachievements_getRank( $profile_user->ID ).' )';
    }
    else{
      $add_ranktext = '';
    }

    echo '<h4>'.__('Rewards', 'wpachievements').$add_ranktext.'</h4>';
    echo '<h5>'.__('Achievements', 'wpachievements').':</h5>';
    echo do_shortcode('[wpa_myachievements user_id="'.$profile_user->ID.'" show_title="" title_class="" image_holder_class="" image_class="achieves_gained_badge" image_width="" achievement_limit=""]' );
    echo '<br /><br /><div class="clear"></div>';

    echo '<h5>'.__('Quests', 'wpachievements').':</h5>';
    echo do_shortcode('[wpa_myquests user_id="'.$profile_user->ID.'" show_title="" title_class="" image_holder_class="" image_class="achieves_gained_badge" image_width="" quest_limit=""]' );
    echo '<br /><br /><br /><br /><div class="clear"></div>';
  }

  /**
  * Show rank status in profile member grid
  *
  * @static
  * @access  public
  * @return
  */
  public static function bp_profile_grid(){

    if ( 'Disable' == wpachievements_get_site_option('wpachievements_rank_status') ) {
      return;
    }

    echo '<div class="item-meta"><strong>'.__('Rank', 'wpachievements').':</strong> '.wpachievements_getRank( bp_get_member_user_id() ).'</div>';
  }

  /**
   * Triggered When a user starts the course
   *
   * @static
   * @access  public
   * @param   int $course_id    Course post ID
   * @param   int $user_id      User ID
   * @return  void
   */
  public static function start_course( $course_id, $user_id ) {
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_start_course',
      'user_id'     => $user_id,
      'post_id'     => $course_id,
      'points'      => (int) wpachievements_get_site_option('wpachievements_wplms_start_course'),
    ) );
  }

  /**
   * Triggered after Course is submitted by user
   *
   * @static
   * @access  public
   * @param   int $course_id    Course post ID
   * @param   int $user_id      User ID
   * @return  void
   */
  public static function submit_course( $course_id, $user_id ) {
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_submit_course',
      'user_id'     => $user_id,
      'post_id'     => $course_id,
      'points'      => (int) wpachievements_get_site_option('wpachievements_wplms_submit_course'),
    ) );    
  }

  /**
   * Triggered after quiz is started-
   *
   * @static
   * @access  public
   * @param   int $quiz_id            Quiz post ID
   * @param   int $user_id            User ID
   * @return  void
   */
  public static function start_quiz( $quiz_id, $user_id ) {
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_start_quiz',
      'user_id'     => $user_id,
      'post_id'     => $quiz_id,
      'points'      => (int) wpachievements_get_site_option('wpachievements_wplms_start_quiz'),
    ) );
  }

  /**
   * Triggered after course is suscribed
   *
   * @static
   * @access  public
   * @param   int $course_id          Course post ID
   * @param   int $user_id            User ID
   * @param   int $group_id
   * @param   array $args

   * @return  void
   */
  public static function course_subscribed( $course_id, $user_id, $group_id = null, $args = null ){
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_course_subscribed',
      'user_id'     => $user_id,
      'post_id'     => $course_id,
      'points'      => (int) wpachievements_get_site_option('wpachievements_wplms_course_subscribed'),
    ) );
  }

  /**
   * Triggered When a user reset the course
   *
   * @static
   * @access  public
   * @param   int $course_id    Course post ID
   * @param   int $user_id      User ID
   * @return  void
   */
  public static function course_reset( $course_id, $user_id ){
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_course_reset',
      'user_id'     => $user_id,
      'post_id'     => $course_id,
      'points'      => (int) wpachievements_get_site_option('wpachievements_wplms_course_reset'),
    ) ); 
  }

  /**
   * Triggered When a user retake the course
   *
   * @static
   * @access  public
   * @param   int $course_id    Course post ID
   * @param   int $user_id      User ID
   * @return  void
   */
  public static function course_retake( $course_id, $user_id ){
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_course_retake',
      'user_id'     => $user_id,
      'post_id'     => $course_id,
      'points'      => (int) wpachievements_get_site_option('wpachievements_wplms_course_retake'),
    ) );    
  }

  /**
   * Triggered When a user review the course
   *
   * @static
   * @access  public
   * @param   int $course_id          Course post ID
   * @param   int $rating
   * @param   string $title
   * @return  void
   */
  public static function course_review( $course_id, $rating, $title ){

    $user_id = get_current_user_id();

    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_course_review',
      'user_id'     => $user_id,
      'post_id'     => $course_id,
      'points'      => (int) wpachievements_get_site_option('wpachievements_wplms_course_review'),
    ) );    
 }

  /**
   * Triggered When a user unsubsribe the course
   *
   * @static
   * @access  public
   * @param   int $course_id    Course post ID
   * @param   int $user_id      User ID
   * @param   int $group_id
   * @return  void
   */
  public static function course_unsubscribe( $course_id, $user_id, $group_id = null ){
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_course_unsubscribe',
      'user_id'     => $user_id,
      'post_id'     => $course_id,
      'points'      => (int) wpachievements_get_site_option('wpachievements_wplms_course_unsubscribe'),
    ) );    
  }

  /**
   * Triggered When a user renew the course
   *
   * @static
   * @access  public
   * @param   int $course_id    Course post ID
   * @param   int $user_id      User ID
   * @return  void
   */
  public static function renew_course( $course_id, $user_id ){
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_renew_course',
      'user_id'     => $user_id,
      'post_id'     => $course_id,
      'points'      => (int) wpachievements_get_site_option('wpachievements_wplms_renew_course'),
    ) );    
  }

  /**
   * Triggered after unit is completed
   *
   * @static
   * @access  public
   * @param   int $unit_id            Unit post ID
   * @param   int $course_progress    Actual Progress of course
   * @param   int $course_id            Course post ID
   * @param   int $user_id            User ID
   * @return  void
   */
  public static function unit_complete( $unit_id, $course_progress, $course_id, $user_id ) {
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_unit_complete',
      'user_id'     => $user_id,
      'post_id'     => $course_id,
      'points'      => (int) wpachievements_get_site_option('wpachievements_wplms_unit_complete'),
    ) );    
  }

  /**
   * Triggered When a user earned a badge
   *
   * @static
   * @access  public
   * @param   int $course_id    Course post ID
   * @param   int $badges
   * @param   int $user_id      User ID
   * @param   int $badge_filter
   * @return  void
   */
  public static function badge_earned( $course_id, $badges, $user_id, $badge_filter ){
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_badge_earned',
      'user_id'     => $user_id,
      'post_id'     => $course_id,
      'points'      => (int) wpachievements_get_site_option('wplms_badge_earned'),
    ) );    
  }

  /**
   * Triggered When a user earned a certificate
   *
   * @static
   * @access  public
   * @param   int $course_id    Course post ID
   * @param   int $pass
   * @param   int $user_id      User ID
   * @param   int $passing_filter
   */
  public static function certificate_earned( $course_id, $pass, $user_id, $passing_filter ){
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_certificate_earned',
      'user_id'     => $user_id,
      'post_id'     => $course_id,
      'points'      => (int) wpachievements_get_site_option('wplms_certificate_earned'),
    ) );        
  }

  /**
   * Triggered after quiz is evaluated-
   *
   * @static
   * @access  public
   * @param   int $quiz_id            Quiz post ID
   * @param   int $marks              Actual Progress of quiz
   * @param   int $user_id            User ID
   * @param   int $max                Max Progress of quiz
   * @return  void
   */
  public static function evaluate_quiz( $quiz_id, $marks, $user_id, $max ) {
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_evaluate_quiz',
      'user_id'     => $user_id,
      'post_id'     => $quiz_id,
      'points'      => (int) wpachievements_get_site_option('wpachievements_wplms_evaluate_quiz'),
    ) );        
  }

  /**
   * Triggered after course is evaluated-
   *
   * @static
   * @access  public
   * @param   int $course_id          Course post ID
   * @param   int $marks              Actual Progress of course
   * @param   int $user_id            User ID
   * @return  void
   */
  public static function evaluate_course( $course_id, $marks, $user_id ) {
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_evaluate_course',
      'user_id'     => $user_id,
      'post_id'     => $course_id,
      'points'      => (int) wpachievements_get_site_option('wpachievements_wplms_evaluate_course'),
    ) );            
  }

  /**
   * Triggered after quiz is submitted by user
   *
   * @static
   * @access  public
   * @param   int $quiz_id    Quiz post ID
   * @param   int $user_id    User ID
   * @return  void
   */
  public static function submit_quiz( $quiz_id, $user_id ) {
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_submit_quiz',
      'user_id'     => $user_id,
      'post_id'     => $quiz_id,
      'points'      => (int) wpachievements_get_site_option('wpachievements_wplms_submit_quiz'),
    ) );            
  }

  /**
   * Triggered When a user retake the quiz
   *
   * @static
   * @access  public
   * @param   int $course_id    Course post ID
   * @param   int $user_id      User ID
   * @return  void
   */
  public static function quiz_retake( $quiz_id, $user_id ){
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_quiz_retake',
      'user_id'     => $user_id,
      'post_id'     => $course_id,
      'points'      => (int) wpachievements_get_site_option('wpachievements_wplms_quiz_retake'),
    ) );                
  }

  /**
   * Triggered When a user reset the quiz
   *
   * @static
   * @access  public
   * @param   int $course_id    Course post ID
   * @param   int $user_id      User ID
   * @return  void
   */
  public static function quiz_reset( $quiz_id, $user_id ){
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_quiz_reset',
      'user_id'     => $user_id,
      'post_id'     => $course_id,
      'points'      => (int) wpachievements_get_site_option('wpachievements_wplms_quiz_reset'),
    ) );                
  }

  /**
   * Triggered after assignment is submitted
   *
   * @static
   * @access  public
   * @param   int $assignment_id    Assignment post ID
   * @param   int $user_id          User ID
   * @return  void
   */
  public static function submit_assignment( $assignment_id, $user_id ) {
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_submit_assignment',
      'user_id'     => $user_id,
      'post_id'     => $assignment_id,
      'points'      => (int) wpachievements_get_site_option('wpachievements_wplms_submit_assignment'),
    ) );                
  }

  /**
   * Triggered after assignment is started
   *
   * @static
   * @access  public
   * @param   int $assignment_id    assignment post ID
   * @param   int $user_id          User ID
   * @return  void
   */
  public static function start_assignment( $assignment_id, $user_id ) {
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_start_assignment',
      'user_id'     => $user_id,
      'post_id'     => $assignment_id,
      'points'      => (int) wpachievements_get_site_option('wpachievements_wplms_start_assignment'),
    ) );       
  }

  /**
   * Triggered after assignment is evaluated
   *
   * @static
   * @access  public
   * @param   int $comment_post_ID    Comment post ID
   * @param   int $value              Value in %
   * @param   int $user_id            Comment User ID
   * @return  void
   */
  public static function evaluate_assignment( $comment_post_ID, $value, $comment_user_id ) {
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'wplms_evaluate_assignment',
      'user_id'     => $comment_user_id,
      'post_id'     => $comment_post_ID,
      'points'      => (int) wpachievements_get_site_option('wpachievements_wplms_evaluate_assignment'),
    ) );       
  }

  /**
   * Register admin section
   *
   * @static
   * @access  public
   * @param   array $sections Array of registered sections for admin settings
   * @return  array           Sections
   */
  public static function register_section( $sections ) {
    $sections['wplms'] = __("WPLMS", 'wpachievements' );

    return $sections;
  }

  /**
   * Admin settings
   *
   * @static
   * @access  public
   * @param   array $default_settings
   * @param   string $short_name
   * @param   string $section
   * @return  array
   */
  public static function admin_settings( $default_settings, $short_name, $section ) {

    if ( $section !== 'wplms' ) {
      return $default_settings;
    }

    $settings = array();
    $settings[] = array( 'title' => __( 'WPLMS', 'wpachievements' ), 'type' => 'title', 'desc' => '', 'id' => 'wplms_options' );

    $settings[] = array(
      'title'   => __( 'User started course', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user started the course.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_start_course',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User submit finished course', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user submitted a finished course.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_submit_course',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'The course was evaluated', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a course was evaluated.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_evaluate_course',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'The user subscribed a course', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a course was subscribed.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_course_subscribed',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'The course was resetted', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a course was resetted.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_course_reset',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'The user retaken a course', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a course was retaken.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_course_retake',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'The user reviewed a course', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a course was reviewed.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_course_review',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'The user unsubscribed a course', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a course was unsubscribed.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_course_unsubscribe',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'The course was renewed', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a course was renewed.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_renew_course',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User completes unit', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user marks unit as complete.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_unit_complete',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User earned a certificate', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user got a certificate in course.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_certificate_earned',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User earned a badge', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user got a badge in the cours.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_badge_earned',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User starts quiz', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user started a quiz.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_start_quiz',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User submits quiz', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user submitted a finished quiz.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_submit_quiz',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'The quiz was evaluated', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a quiz was evaluated.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_evaluate_quiz',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'The user retaken a quiz', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a quiz was retaken.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_quiz_retake',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'The quiz was resetted', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a quiz was resetted.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_quiz_reset',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User starts assignment', 'wpachievements' ),
      'desc'    => __( 'Points awarded when an assignment was started.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_start_assignment',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User submits assignment', 'wpachievements' ),
      'desc'    => __( 'Points awarded when an assignment was submitted.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_submit_assignment',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User evaluated assignment', 'wpachievements' ),
      'desc'    => __( 'Points awarded when an assignment was evaluated.', 'wpachievements' ),
      'id'      => $short_name.'_wplms_evaluate_assignment',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array( 'type' => 'sectionend', 'id' => 'Sensei_options');

    return $settings;
  }

  /**
   * Return all available triggers with descriptions
   *
   * @static
   * @access  public
   * @return  array Array of triggers and descriptions
   */
  public static function get_triggers() {
    return array(
      "wplms_start_course"        => __('The user started the course', 'wpachievements'),
      "wplms_submit_course"       => __('The user submitted a finished course', 'wpachievements'),
      "wplms_evaluate_course"     => __('The course was evaluated', 'wpachievements'),
      "wplms_course_subscribed"   => __('The user suscribed the course', 'wpachievements'),
      "wplms_course_reset"        => __('The course was resetted', 'wpachievements'),
      "wplms_course_retake"       => __('The user retaken the course', 'wpachievements'),
      "wplms_course_review"       => __('The user reviewed the course', 'wpachievements'),
      "wplms_course_unsubscribe"  => __('The user unsuscribed the course', 'wpachievements'),
      "wplms_renew_course"        => __('The course was renewed', 'wpachievements'),
      "wplms_unit_complete"       => __('The user marks unit as complete', 'wpachievements'),
      "wplms_certificate_earned"  => __('The user earned a certificate in course', 'wpachievements'),
      "wplms_badge_earned"        => __('The user earned a badge in the course', 'wpachievements'),
      "wplms_start_quiz"          => __('The user started a quiz', 'wpachievements'),
      "wplms_submit_quiz"         => __('The user submitted a finished quiz', 'wpachievements'),
      "wplms_evaluate_quiz"       => __('The quiz was evaluated', 'wpachievements'),
      "wplms_quiz_retake"         => __('The user retaken a quiz', 'wpachievements'),
      "wplms_quiz_reset"          => __('The quiz was resetted', 'wpachievements'),
      "wplms_submit_assignment"   => __('The assignment was submitted', 'wpachievements'),
      "wplms_start_assignment"    => __('The assignment was started', 'wpachievements'),
      "wplms_evaluate_assignment" => __('The assignment was evaluated', 'wpachievements'),
    );
  }

  /**
   * Register trigger events displayed on the achievements page
   *
   * @static
   * @access  public
   * @return  string
   */
  public static function trigger_dropdown() {
    ob_start();
    ?>
    <optgroup label="WPLMS Events">
      <?php foreach ( self::get_triggers() as $key => $description ) : ?>
        <option value="<?php echo $key; ?>"><?php echo $description; ?></option>
      <?php endforeach; ?>
    </optgroup>
    <?php
    ob_end_flush();
  }

  /**
   * Return description of selected trigger
   *
   * @static
   * @access  public
   * @param   string $trigger Trigger key
   * @return  string          Trigger description
   */
  public static function selected_trigger( $trigger ) {
    $triggers = self::get_triggers();

    if ( array_key_exists( $trigger, $triggers ) ) {
      return $triggers[ $trigger ];
    }

    return $trigger;
  }

  /**
   * Generate activity descriptions
   *
   * @static
   * @access  public
   * @param   string $text  Activity description
   * @param   string $type  Activity trigger
   * @param   int $points   Points
   * @param   int $times    Occurrences
   * @return  string        Activity description
   */
  public static function activity_description( $text, $type, $points, $times ) {

    switch( $type ) {
      case 'wplms_start_course': {
        $text = sprintf( _n( 'for starting %s course', 'for starting %s courses', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_submit_course': {
        $text = sprintf( _n( 'for submitting %s finished course', 'for submitting %s finished courses', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_evaluate_course': {
        $text = sprintf( _n( 'for evaluating %s course', 'or evaluating %s courses', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_course_subscribed': {
        $text = sprintf( _n( 'for subscribing %s course', 'or subscribing %s courses', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_course_reset': {
        $text = sprintf( _n( 'for resetting %s course', 'or resetting %s courses', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_course_retake': {
        $text = sprintf( _n( 'for retaking %s course', 'or retaking %s courses', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_course_review': {
        $text = sprintf( _n( 'for reviewing %s course', 'or reviewing %s courses', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_course_unsubscribe': {
        $text = sprintf( _n( 'for unsubscribing %s course', 'or unsubscribing %s courses', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_renew_course': {
        $text = sprintf( _n( 'for renewing %s course', 'or renewing %s courses', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_unit_complete': {
        $text = sprintf( _n( 'for marking %s unit as complete', 'for marking %s units as complete', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_certificate_earned': {
        $text = sprintf( _n( 'for earning %s certificate', 'for earning %s certificates', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_badge_earned': {
        $text = sprintf( _n( 'for earning %s badge', 'for earning %s badges', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_start_quiz': {
        $text = sprintf( _n( 'for starting %s quiz', 'for starting %s quizzes', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_submit_quiz': {
        $text = sprintf( _n( 'for submitting %s quiz', 'for submitting %s quizzes', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_evaluate_quiz': {
        $text = sprintf( _n( 'for evaluating %s quiz', 'for evaluating %s quizzes', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_quiz_retake': {
        $text = sprintf( _n( 'for retaking %s course', 'or retaking %s courses', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_quiz_reset': {
        $text = sprintf( _n( 'for resetting %s course', 'or resetting %s courses', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_submit_assignment': {
        $text = sprintf( _n( 'for submitting %s assignment', 'for submitting %s assignments', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_start_assignment': {
        $text = sprintf( _n( 'for starting %s assignment', 'for starting %s assignments', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_evaluate_assignment': {
        $text = sprintf( _n( 'for evaluating %s assignment', 'for evaluating %s assignments', $times, 'wpachievements' ), $times );
      } break;
    }

    return $text;
  }

  /**
   * Generate quest step descriptions
   *
   * @static
   * @access  public
   * @param   string $text  Step description
   * @param   string $type  Trigger type
   * @param   int $times    Occurrences
   * @return  string        Step description
   */
  public static function quest_description( $text, $type, $times ) {

    switch( $type ) {
      case 'wplms_start_course': {
        $text = sprintf( _n( 'The user started %s course', 'he user started %s courses', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_submit_course': {
        $text = sprintf( _n( 'The user submitted %s finished course', 'The user submitted %s finished courses', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_evaluate_course': {
        $text = sprintf( _n( '%s course was evaluated', '%s courses were evaluated', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_course_subscribed': {
        $text = sprintf( _n( 'The user subscribed %s finished course', 'The user subscribed %s finished courses', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_course_reset': {
        $text = sprintf( _n( 'The course was resetted %s time', 'The courses were resetted %s times', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_course_retake': {
        $text = sprintf( _n( 'The user retaken %s finished course', 'The user retaken %s finished courses', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_course_review': {
        $text = sprintf( _n( 'The user reviewed %s finished course', 'The user reviewed %s finished courses', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_course_unsubscribe': {
        $text = sprintf( _n( 'The user unsubscribed %s finished course', 'The user unsubscribed %s finished courses', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_renew_course': {
        $text = sprintf( _n( 'The course was %s renewed', 'The courses were %s renewed', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_unit_complete': {
        $text = sprintf( _n( 'The user marks %s unit as complete', 'The user marks %s units as complete', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_certificate_earned': {
        $text = sprintf( _n( 'The user earned %s Certificate.', 'The user earned %s Certificates.', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_badge_earned': {
        $text = sprintf( _n( 'The user earned %s Badge.', 'The user earned %s Badges.', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_start_quiz': {
        $text = sprintf( _n( 'The user started %s quiz', 'The user started %s quizzes', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_submit_quiz': {
        $text = sprintf( _n( 'The user submitted %s finished quiz', 'The user submitted %s finished quizzes', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_evaluate_quiz': {
        $text = sprintf( _n( '%s quiz was evaluated', '%s quizzes were evaluated', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_quiz_retake': {
        $text = sprintf( _n( 'The user retaken %s finished quiz', 'The user retaken %s finished quizzes', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_quiz_reset': {
        $text = sprintf( _n( 'The quiz was resetted %s time', 'The quizzes were resetted %s times', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_start_assignment': {
        $text = sprintf( _n( '%s assignment was started', '%s assignments were started', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_submit_assignment': {
        $text = sprintf( _n( '%s assignment was submitted', '%s assignments were submitted', $times, 'wpachievements' ), $times );
      } break;

      case 'wplms_evaluate_assignment': {
        $text = sprintf( _n( '%s assignment was evaluated', '%s assignments were evaluated', $times, 'wpachievements' ), $times );
      } break;
    }

    return $text;
  }

  /**
  * Filters the show text which controls visibility of post ID inpuz
  *
  * @static
  * @access  public
  * @param   string $show          Show String to enable view
  * @param   string $cur_post      Current Post
  * @param   string $cur_trigger   Current Trigger
  * @return  string                Show String to enable view
  */
  public static function event_details_data_post_id_show( $show, $cur_post, $cur_trigger ) {

    if ( !empty($cur_post) && ( $cur_trigger == 'wplms_start_course' || $cur_trigger == 'wplms_submit_course' || $cur_trigger == 'wplms_evaluate_course' || $cur_trigger == 'wplms_unit_complete' || $cur_trigger == 'wplms_start_quiz' || $cur_trigger == 'wplms_submit_quiz' || $cur_trigger == 'wplms_evaluate_quiz' || $cur_trigger == 'wplms_submit_assignment' || $cur_trigger == 'wplms_start_assignment' || $cur_trigger == 'wplms_evaluate_assignment'  || $cur_trigger == 'wplms_course_subscribed' || $cur_trigger == 'wplms_course_reset' || $cur_trigger == 'wplms_course_retake' || $cur_trigger == 'wplms_course_review' || $cur_trigger == 'wplms_course_unsubscribe' || $cur_trigger == 'wplms_renew_course' || $cur_trigger == 'wplms_quiz_retake' || $cur_trigger == 'wplms_quiz_reset' ) )
    {
      $show = ' style="display:block !important;"';
    }
    return $show;
  }

  /**
  * Filters the text of postid input label
  *
  * @static
  * @access  public
  * @param   string $postid_title  Title of the postid field
  * @param   string $cur_post      Current Post
  * @param   string $cur_trigger   Current Trigger
  * @return  string                Title
 */
  public static function event_details_data_post_id_text( $postid_title, $cur_post, $cur_trigger ) {

    if ( !empty($cur_post) && ( $cur_trigger == 'wplms_start_course' || $cur_trigger == 'wplms_submit_course' || $cur_trigger == 'wplms_evaluate_course' || $cur_trigger == 'wplms_unit_complete' || $cur_trigger == 'wplms_start_quiz' || $cur_trigger == 'wplms_submit_quiz' || $cur_trigger == 'wplms_evaluate_quiz' || $cur_trigger == 'wplms_submit_assignment' || $cur_trigger == 'wplms_start_assignment' || $cur_trigger == 'wplms_evaluate_assignment' || $cur_trigger == 'wplms_certificate_earned' || $cur_trigger == 'wplms_certificate_earned' || $cur_trigger == 'wplms_course_subscribed'  || $cur_trigger == 'wplms_course_reset' || $cur_trigger == 'wplms_course_retake' || $cur_trigger == 'wplms_course_review' || $cur_trigger == 'wplms_course_unsubscribe' || $cur_trigger == 'wplms_renew_course' || $cur_trigger == 'wplms_quiz_retake' || $cur_trigger == 'wplms_quiz_reset' ) )
    {
      switch ($cur_trigger) {
        case 'wplms_start_course':
        case 'wplms_submit_course':
        case 'wplms_evaluate_course':
        case 'wplms_certificate_earned':
        case 'wplms_badge_earned':
        case 'wplms_course_subscribed':
        case 'wplms_course_reset':
        case 'wplms_course_retake':
        case 'wplms_course_review':
        case 'wplms_course_unsubscribe':
        case 'wplms_renew_course':
            $postid_title = __('Course ID', 'wpachievements');
            break;
        case 'wplms_start_quiz':
        case 'wplms_submit_quiz':
        case 'wplms_evaluate_quiz':
        case 'wplms_quiz_retake':
        case 'wplms_quiz_reset':
            $postid_title = __('Quiz ID', 'wpachievements');
            break;
        case 'wplms_start_assignment':
        case 'wplms_submit_assignment':
        case 'wplms_evaluate_assignment':
            $postid_title = __('Assignment ID', 'wpachievements');
            break;
        case 'wplms_unit_complete':
            $postid_title = __('Unit ID', 'wpachievements');
            break;
      }
    }
    return $postid_title;
  }

  /**
   * Check if we have a valid activity
   *
   * @param   bool $proceed True on valid activity
   * @param   string $type
   * @param   int $uid
   * @param   int $postid
   * @param   int $ach_ID
   * @return  bool
   */
  public static function check_actvity( $proceed, $type, $ach_ID, $postid, $uid ) {

    // evaluate Course, Quiz,
    if ( 'wplms_evaluate_course' == $type || 'wplms_evaluate_quiz' == $type) {

      $current_wplms_evaluate_limit = intval( get_post_meta( $ach_ID, '_achievement_wplms_evaluate_limit', true ) );
      if ( ( $current_wplms_evaluate_limit > 0 ) && ( $current_wplms_evaluate_limit <= 100 ) ) {
        $current_wplms_mark = intval( get_post_meta( $postid, $uid, true ) );
        switch( $type ) {
          case 'wplms_evaluate_course':
                if ( $current_wplms_mark && ( $current_wplms_mark <= $current_wplms_evaluate_limit) ) $proceed = false;
                break;
          case 'wplms_evaluate_quiz':
                if ( $current_wplms_mark ) {
                  // Get all points for the max amount
                  $questions = bp_course_get_quiz_questions($postid,$uid);
                  $max_points= array_sum($questions['marks']);
                  $correct_answers_ratio_in_percent = $current_wplms_mark / $max_points * 100;
                  if ( $correct_answers_ratio_in_percent <= $current_wplms_evaluate_limit) $proceed = false;
                }
                break;
        }
      }
    }
    WPAchievements()->logger()->add( 'log', __FUNCTION__ . " - wpachievements_wplms_check_actvity: {$type}, user: {$uid}, post_id: {$postid}, ach_ID: {$ach_ID}, proceed: {$proceed}" );
    return $proceed;
  }

  /**
   * Handle achievements_event_details_footermeta box
   *
   * @param   int $post->ID
   * @param   int $cur_trigger_id
   * @param   int $cur_trigger
   * @param   int $cur_trigger_desc
   * @return  void
   */
  public static function achievements_event_details_footer( $post_id, $cur_trigger_id, $cur_trigger, $cur_trigger_desc ){

      $cur_wplms_evaluate_limit = intval( get_post_meta( $post_id, '_achievement_wplms_evaluate_limit', true ) );

      if ( !empty( $cur_wplms_evaluate_limit ) && ($cur_wplms_evaluate_limit > 0 ) ) {
        $show = ' style="display:block !important;"';
      }
      else {
        $show = '';
      }

      echo '<span id="wplms_evaluate_limit"'.$show.'>
      <label for="wpachievements_achievement_wplms_evaluate_limit">'.__('Limit to pass', 'wpachievements').': <small>(Optional)</small></label>
      <div class="spinner-holder">
        <div style="position:relative;">
          <span id="wpachievements_achievement_wplms_evaluate_limit">%</span>
          <input type="number" min="1" id="wpachievements_achievement_wplms_evaluate_limit" name="wpachievements_achievement_wplms_evaluate_limit" value="'.$cur_wplms_evaluate_limit.'" />
        </div>
      </div>
      </span>';
  }
}
WPAchievements_Module_WPLMS::init();

endif;